function Surtseyan_bombs_code_variouskphi
%% Uses the method of lines to solve nondim Surtseyan bombs model
% Emma Greenbank & Mark McGuinness
%
% A coordinate tranformation is used to transform a variable mesh to a uniform
% mesh. In the original (frozen) coordinates, the mesh is finer next to the
% flash point, where the most significant gradients are seen.
% Permeability depends on porosity as fitted to data.  
% Simulations are currently set to stop when pressure
% in magma next to flash point reaches a maximum.
% 
% A ramped initial temperature based on the erf similarity solution is used, 
% but a stepped initial pressure, set at flash to match the temperature there,
% but at atmospheric elsewhere in the magma, so that density will
% be set by the values of P, T in that region as it is elsewhere in the magma.
% The default is to set the RampDistance to 1E-05/R2, that is, ten
% microns, being a typical pore size for pumice. Also to set the smallest mesh
% size to be about 1/10 of this for numerical resolution. 
% Best to use at least 800 mesh points in magma for accuracy.
%
% Results are also compared with the prediction of equation (6.10) in paper
% This file is designed to reproduce Figs. 4(f), 6(a), and 7 in the arXiv
% manuscript.

global T0 R2 R1 s0 p2 Tnp1 Tn Tm2 Tm3 delta5 epsilon3 RampDistance pmaxtheor tmaxtheor pmaxtheorACC tmaxtheorACC

% input log10(permeability) versus porosity data 

IanData=[0.240	-12.53;  % porosity,  log10 k
0.306	-12.38;
0.318	-12.11;
0.338	-11.71;
0.345	-11.89;
0.350	-11.58;
0.377	-11.70;
0.415	-11.30;
0.422	-11.84;
0.436	-11.28;
0.466	-11.29;
0.478	-10.75;
0.506	-10.83;
0.512	-10.51;
0.525	-10.48;
0.560	-10.82;
0.586	-10.41];

LogPerm=IanData(:,2); poro=IanData(:,1); % select and name data columns
NDat=length(LogPerm);
AugPoro=ones(NDat,2); 
AugPoro(:,2) = poro; % augment porosities with ones in second column

% fit a line through this data
Coeffs=AugPoro\LogPerm; % the ones allow an intercept to be fitted
disp('fit of permeability to porosities:')
display(Coeffs)  % these are -14.07 and 6.40 resp, for Ian's data
                 % that is, intercept -14.07 and slope 6.40
              
%chporo=linspace(0.1, 0.95, 11);  % choose the porosities that will be used in these simulations
chporo=linspace(0.2, 0.6, 5);
Augchp=ones(length(chporo),2);   % create Augchp
Augchp(:,2)=chporo; % put the spaced porosities into 2nd col of Augchp

ChpFit=Augchp*Coeffs;  % permeability k fitted to data, for porosities ranging 
                       % over chporo values

figno=65; defFont=18;
figure(figno) % plot data input, and fitted line
clf('reset')
plot(poro,LogPerm,'or','LineWidth', 3,'MarkerSize',20) % data
hold on;
plot(chporo, ChpFit,'-b','LineWidth', 2,'MarkerSize',15)
hold off;
ax=gca;
ax.FontSize=defFont; ax.FontName='Times'; 
ylabel('log10 k');xlabel('\phi');

tic

%m=800; % recommended number of T and rho variables in magma, 2m in total are used, plus n in slurry
m=100; % for quick test runs; not accurate enough really, recommend 800 here
n=m/2; % recommended slurry mesh size; then get accurate a in slurry, called aas

% set tolerances for simulations, and events to ensure stop simulations
% once a maximum flash pressure is encountered (timewise):
options=odeset('RelTol',1.0e-8,'AbsTol',1.0e-8,'events',@MyEvents);

kset=10.^ChpFit; % the permeability values to be used for simulations
nitns=length(kset); % number of iterations on k, phi

pmaxvals=zeros(1,nitns);
pmaxtheorV=zeros(1,nitns);
pmaxtheorVACC=zeros(1,nitns);
tmaxvals=pmaxvals;
tmaxtheorV=pmaxtheorV;
tmaxtheorVACC=zeros(1,nitns);

tmax=0.01; % maximum allowed dimensionless time. It is 1 when all slurry is evaporated.
nft=501; finet = zeros(nft,1);
figwarn=40; % figure number to be used for warnings

for ik=1:nitns  %use counter ik for main iteration loop, varying the permeability k
    
disp({ik, ' of ', nitns})   
    
k=kset(ik);
phi=chporo(ik); % set porosity and k consistently with data from Ian


%%
% use a transform to refine mesh near flash point. Use an evenly (unit)
% spaced variable chi for slurry ranging from 0 to n+1 in value, so there
% are n+2 indices in total, n ode variables plus two boundary values. 
% Use psi for hot magma ranging from 0 to m+1 in value, unit spaced like
% chi, with m variables solved for in odes, and two boundary values.

RescaleSurtsey;  % generates epsilon and delta values, and pmaxtheor, tmaxtheor.

% RampDistance is set in the previous call to RescaleSurtsey
% It is the nondimensional distance over which initial temperature ramps
% from slurry temperature to hot magma temperature

smallestMesh=RampDistance/10; % the smallest mesh size to be used
                              % for accuracy we recommend 1/10 the ramp
                              % distance
% initialise storage arrays:
pmaxtheorV(ik) = pmaxtheor;  % to hold theoretical values of pmax at these parameter values
tmaxtheorV(ik) = tmaxtheor;  % theoretical values of time to reach pmax 
pmaxtheorVACC(ik) = pmaxtheorACC;
tmaxtheorVACC(ik) = tmaxtheorACC;


tSince = RampDistance^2/delta5/16; % time since emplacement and step function initial conditions

ximin=smallestMesh/(1-s0);  % s0=R1/R2 is the initial dimless value of s
zetamin = smallestMesh/s0;    % the smallest mesh in zeta                       
                        
% now set aam, given m, so that the smallest mesh size in the magma is roughly smallestMesh:
% set up the mesh size in the magma, using the specified m. 
% find an approximate value for aam that ensures the mesh is not
% too small near the flash point:
% 
aam = 2/(pi*(m+1)*(m+1)*ximin);  % this is only approximate, and only in magma

myGetAAmag = @GetZmag; %the zero of this function gives an accurate "a" value
aam = fzero(myGetAAmag,[aam/2 aam*2]); % an accurate value for aam, in magma

% get a different aam value for slurry, having first chosen a value for n
aas=2/(pi*(n+1)*(n+1) * zetamin); %this is only a rough estimate, to help

myGetAAslurry = @GetAAslurry; %the zero of this function gives an accurate "a" value
aas = fzero(myGetAAslurry,[aas/2 aas*2]); % an accurate value for aam in slurry
                        
Bm = atan(aam*(m+1)); Bn = atan(aas*(n+1)); %constants needed later

chi=linspace(0,n+1,n+2); % 0,1,2,3, ... n+1 in value
psi=linspace(0,m+1,m+2); % 0,1,2,3, ... m+1 in value
zeta=atan(aas*chi)/atan(aas*(n+1));  % zeta in slurry goes from 0 to 1, tightens on 1
xi= 1 + atan(aam*(psi-m-1))/atan(aam*(m+1)); % xi in hot magma goes from 0 to 1, 
                                           % and tightens mesh on 0
Cm = Bm/aam*(1+aam*aam*(psi-m-1).*(psi-m-1)); % vector term that arises from chain rule

Cn = Bn/aas*(1+aas*aas* chi.*chi); % vector term that arises from chain rule                                          

if (xi(2)-xi(1)) < smallestMesh/2   % possibly n is too large, smallest spacing is too small
    disp('bailing out, smallest mesh size is too small, maybe n is too large?')
    disp(xi(1:3))
    display(StopNow)
end

y0=InitialConds(n,m);  % set up the initial conditions

    s = y0(n+2*m+1); % flash location fed into FlashBC as global
    Tm2 = y0(n+1); % leftmost temperature in magma, needed by FlashBC, global
    Tm3 = y0(n+2);  % next temperature, also needed for 2nd order first difference approxn to gradient
    p2= y0(n+m+1)*Tm2 ;   % needed by FlashBC, passed as global
    p3=y0(n+m+2)*Tm3; % also needed by FlashBC
    Tnp1 = y0(n); % temperature in slurry next to flash, needed by FlashBC, global
    Tn= y0(n-1);  % temperature in slurry, two nodes from flash, needed by FlashBC, global

%======================================================================
% solve the problem, using current parameter values:

soln=ode15s(@SurtseyModel,[0 tmax], y0, options); % use a stiff solver

tmaxt = 0.9999*soln.x(end);  % the last time value returned by ode15s 
                             % can be a fraction too large, so pull back
tmin = tmaxt/1E05;
finet(2:end) = logspace(log10(tmin),log10(tmaxt),nft-1); %finer times, log spaced, for s, pflash, tflash vs time

display(k)
display(phi)

FineVals = deval(soln,finet); % get solution at finer spaced times finet

% note that deval returns each time in a different column.
% Unpacksolutions fails if no of rows = no of columns AND different times
% are in different columns. It wants different times on diff rows.
% so ensure that FineVals and CoarseVals have diff times on diff rows: 
[~,Tmagf,rhomagf,~] = UnpackSolutions(FineVals'); %gets T, rho, s from solutions
                                                  % finely spaced in time
pmagf = rhomagf.*Tmagf;

% watch out for premature finish:

if (tmax- soln.x(end)) < 1E-05*tmax
    disp('warning Will Robinson:')
    disp('did not run long enough')
    disp('max run time tmax is set to: ')
    disp(tmax)
    disp('I advise you increase tmax')
    figwarn=figwarn+1;
    figure(figwarn)
    clf('reset')
    plot(finet,pmagf(:,1),'-k','LineWidth', 2) % look at pressure values vs time at flash
    ax=gca;
    ax.FontSize=defFont; ax.FontName='Times'; 
    ylabel('Pf');xlabel('t');   
end


[pfmax, ipm] = max(pmagf(:,1)); % find the maximum of pressure at flashing front, and its index
tpfmax = finet(ipm); % the time at which p was a max

% simulated values for max p, time of max
pmaxvals(ik) = pfmax;
tmaxvals(ik) =tpfmax;

display(pfmax)

end

% plot results, pmax vs k, together with the tensile strength
% This gives Fig 4(f) in our paper

figno=1; defFont=18;

figure(figno)  % pmax vs k, simulated, together with tensile strength
       % corrected for porosity (dashed line)
       % and a vertical dotted line at the smallest k seen in data
clf('reset') % clear figure settings if any
plot(log10(kset),pmaxvals,'-or','LineWidth', 2,'MarkerSize',15) % check out Pmax
hold on;
plot(log10(kset), (1-chporo)*20, '--k','LineWidth', 4)  % this is a set
                                            % of threshold p values, based on
                                            % 20(1-phi) tensile strength, in atmospheres
                                            % (i.e. nondimensional pressure)
plot([-12.5, -12.5],[min(pmaxvals), max(pmaxvals)],':b','LineWidth',5); % smallest k seen in data
hold off;
ax=gca;
ax.FontSize=defFont; ax.FontName='Times'; 
ylabel('Pmax');xlabel('log k');

Aratio=pmaxvals./pmaxtheorV;  % get a simple fit to simulated Pmax values
Avratio = mean(Aratio);


% a log-log version of pressures, with theoretical upper bound on Pmax values (simple only):
% this gives Fig 6(a) plus an extra line for the fitted straight line which
% is in Fig 7:
figno=figno+1;
figure(figno)
clf('reset')
plot(log10(kset),log10(pmaxtheorV),'-b','LineWidth', 4) % theoretical upper limit to Pmax
hold on;
plot(log10(kset),log10(Avratio*pmaxtheorV),'-k','LineWidth', 4) % fitted Pmax
plot(log10(kset),log10(pmaxvals),'-or','LineWidth', 3,'MarkerSize',15) % simulated Pmax
plot(log10(kset), log10( (1-chporo)*20), '--k','LineWidth', 4)  % this is a more careful set
                                            % of threshold (tensile strength) values, based on
                                            % 20(1-phi)
plot([-12.5, -12.5],[4, -1],':b','LineWidth',5); % minimum observed k in data
hold off;
ax=gca;
ax.FontSize=defFont; ax.FontName='Times'; 
ylabel('log Pmax');xlabel('log k');

% find ratio of simulated to theoretical Pmax and use average as a fitted
% theoretical behaviour to Pmax. Compare with simulated values:


AlogDiff=log10(pmaxtheorV) - log10(pmaxvals);  % get a simple fit to log10 of simulated Pmax values
AvLOGdiff = mean(AlogDiff);

display(Avratio)
display(AvLOGdiff)

Atratio=tmaxvals./tmaxtheorV;  % get a simple fit to simulated times tmax values
Atvratio = mean(Atratio);

display(Avratio)
display(Atvratio)

toc

    function y0=InitialConds(n,m)
        %% InitialConds is to set up the initial values of the variables

        global p0 Haitch
        
        p0 = 1; %initial pressure
%        Ti=1; % initial temperature in magma if no ramping
%         Tslurry=T0; % inital temperature in slurry if no ramping
%         
%         rhoz=p0/Ti; % initial density of vapour in magma
%         
%         y0=zeros(n+2*m+1,1); % set up variables as column vector
%         
%         y0(1:n) = Tslurry; % the first n are T in slurry
%         y0(n+1:n+m) = Ti;       % the next m are T in magma
%         y0(n+m+1:n+2*m) = rhoz; % the next m are density in magma


        % but rather than a step change across the flash point, I want to 
        % ramp up rapidly from Tslurry to Ti, in the magma, and down from Ts
        % to T0 in the slurry as we move away from the flash point. 
        % The similarity solutions are used over the full distances, but they 
        % effectively ramp over the distance RampDistance in each medium. This
        % distance is the (nondimensional) typical pore size, of the order of the size of a
        % representative elementary volume over which the rock matrix and
        % the fluid in the pores are averaged. RampTimes and RampTimem are 
        % the equivalent times from step form of the error functions, in
        % slurry and in magma respectively.
        % I  need the Tflash value; call UnpackSolutionsInitial for this, 
        % which is based on UnpackSolutions but uses the error function form
        % to compute gradients at flash.
        % then replace all T values with ramping values.
        % I won't ramp the pressures in the magma; the densities will
        % accommodate the resulting pressures and temperatures. Note that
        % the pressure gradient affects the just-calculated Tflash
                           
            p2= p0 ;   % needed by FlashBCinit, passed as global
            p3= p0 ;   % needed by FlashBCinit, passed as global
            
            % check straddle zero. Look between slurry temp and critical
            % temp

            r1=FlashBCinit(0.25);
            r2=FlashBCinit(0.5); %T=0.5 is approximately critical temperature
            if r1*r2 > 0   % same sign, not straddling!!
                disp('not straddling zero?? inside initial conds 1')
                disp(StopNow)
            else % we do have straddle: find initial flash temperature
                Tflash = fzero(@FlashBCinit,[0.25 0.5]); % solve BC at flash for temperature there
            end
            
%             Lookie = LookAtFlashBCInit(Tflash);
%             format shortEng
%             disp('result of look at Flash BC init:')
%             disp(Lookie)
%        
        % Tflash depends on the temperature gradients on either side, and on the P gradient 
        % in the magma. These
        % are known - I have specified the distance over which initial T ramps
        % up, and I know what it ramps to. So Tflash should line up OK.
        
        RNslurr= s0*zeta(2:n+1); % the mesh in the slurry at time zero; nondimensional r values
                                 % the first and last zeta(1:n+2) is at boundary
                                 % values, not y0 values
        RNmag=s0+(1-s0)*(xi(2:m+1));  % the mesh in the magma at time zero, in terms of the 
                         % nondimensional R values, normalised on R2. First
                         % and last values are boundary locations, not y0
 
% start with similarity solution for T in magma and in slurry:
        Tmagma = Tflash + (1-Tflash)*erf((RNmag-s0)/(2*sqrt(delta5*tSince)));
        
        Tslurry = Tflash + (T0-Tflash)*erf( (s0-RNslurr)/(2* sqrt(epsilon3*tSince)));
        
        y0(1:n) = Tslurry; % the first is set by the BC at origin, the last is Tflash
        y0(n+1:n+m) = Tmagma; % the first is Tflash, the last is the boundary value, close to 1
        y0(n+m+1:n+2*m) = p0./Tmagma;        % densities in magma
        y0(n+2*m+1) = s0;  % initial position of flash front s
        
         Pflash = exp(Haitch*(Tflash - T0)./Tflash);
         disp('initial conditions, Pf and Tf, using erf slopes:')
         disp(Pflash)
         disp(Tflash)
         
         % check what the flash conditions will be with these surrounding
         % p, T values:
         Tnp1 = Tslurry(n);
         Tn = Tslurry(n-1);
         Tm2 = Tmagma(1);
         Tm3 = Tmagma(2);
         s=s0;
         
            r1=FlashBC(0.25);
            r2=FlashBC(0.5); %T=0.5 is approximately critical temperature
            if r1*r2 > 0   % same sign, not straddling!!
                disp('not straddling zero?? inside initial conds 2')
                disp(StopNow)
            else % we do have straddle: find initial flash temperature
                Tflash2 = fzero(@FlashBC,[0.25 0.5]); % solve BC at flash for temperature there
            end
          Pflash2 = exp(Haitch*(Tflash2 - T0)./Tflash2);
          disp('recalculated flash conditions using initial conditions, Pf2 and Tf2:')
          disp(Pflash2)
          disp(Tflash2)
 %         disp(StopNow)
          
%           disp('LHS of running Flash BC on init Tf')
%           result=LeftSide(Tflash);
%           disp(result)
% 
%           disp('LHS of running Flash BC on running Tf')
%           result=LeftSide(Tflash2);
%           disp(result)
% 
%           result = -Tm3 +4*Tm2-3*Tflash;
%           disp('this should match second comp of Lookie but uses init Tf')
%           disp(result)
% 
%           result = -Tm3 +4*Tm2-3*Tflash2;
%           disp('this should match second comp of Lookie but uses running Tf')
%           disp(result)
% 
%  
%           result = -Bn/Bm  * aam/aas *(s0-1)/s0 * ...
%         (1+aas*aas*(n+1)*(n+1))/(1+aam*aam*(m+1)*(m+1)) *(3*Tflash-4*Tnp1+Tn);
%           disp('this should match third comp of Lookie, uses init Tf')
%           disp(result)
%           
%           result = -Bn/Bm  * aam/aas *(s0-1)/s0 * ...
%         (1+aas*aas*(n+1)*(n+1))/(1+aam*aam*(m+1)*(m+1)) *(3*Tflash2-4*Tnp1+Tn);
%           disp('this should match third comp of Lookie, uses running Tf')
%           disp(result)
%           
          
        % plot the initial values of T in slurry and magma, and of the densities in
        % the magma (2nd vertical axis)
        
%         figure(33)  % plot initial spacings in nondimensional r values (r/R2)
%         clf('reset')
%         
%         yyaxis left;
%         plot(RNslurr, y0(1:n),'+b')  %T in slurry
%         hold on;
%         plot(s0, Tflash,'or','MarkerSize',15)  % T at flash
%         plot(RNmag, y0(n+1:n+m), 'ok')  % T in magma
%         ylabel('T');xlabel('nondim r');
%         title('Initial Conditions')
%         
%         yyaxis right;
%                 
%         plot(s0, Pflash/Tflash,'dr','MarkerSize',15)  % density at flash
%         plot(RNmag, y0(n+m+1:n+2*m), '+r')
%         hold off;
%         ylabel('\rho');
% 
%         ax=gca;
%         defFont=24;
%         ax.FontSize=defFont; ax.FontName='Times';
%         
%         format
%         
%           disp(StopNow)
     
%         
       
    end

    function [Ts,Tm,rho,ess] = UnpackSolutions(y)
        %% UNPACKSOLUTIONS takes the y values solved for, and unpacks them
        % y unpacks to T values in slurry, T values in magma, rho values in 
        % magma, and flashing front location s(t)
        % values at boundaries are also added
        % I need to ensure this can handle a single vector y, and a matrix
        % with rows or columns of values, one for each of a series of times
        
        global  Haitch  p0 Tsurface
        
        ysize=size(y);  % many rows, 1 column usually if solving. 
 
        nrows = ysize(1);
        ncols = ysize(2); % handles multiple rows/times ok too
        
        % make sure different rows are different times, so the number of columns
        % is n+2m+1, by finding whether
        % rows or columns match the expected length n+2m+1:
        if (nrows == n+2*m+1) && (ncols ~= n+2*m+1)  %uh oh, each column is a time
            y=y';    % switch rows and columns; don't switch if by some chance, 
                     % both the number of times
                     % and the number of variables match up; in that case it is up
                     % to the programmer to ensure different rows are at
                     % different times.

            nrows=ncols; % because you just switched rows and cols. Only need
                         % nrows now
 
        end
        
        % now each row is a different time. There may be only one row:
        
        Ts(:,2:n+1)= y(:,1:n); % temperatures in the slurry, excluding the boundaries
        Tm(:,2:m+1) = y(:,n+1:n+m); % temperatures in the magma, excl boundaries
        rho(:,2:m+1) = y(:,n+m+1:n+2*m); % vapour density in magma
   
        
        % now add the boundary values on to the solved values
        
        pv2 = rho(:,2).*Tm(:,2);  % pressures in magma adjacent to flash, needed for FlashBC
        pv3 = rho(:,3).*Tm(:,3);  % pressures in magma 2 nodes from flash, needed for FlashBC
        Ts(:,1) = Ts(:,2); % no heat flow at origin, in slurry
        ess = y(:,n+2*m+1);    % flash locations
        
        % use a for loop to find the flash temperatures:
        Tflash = zeros(nrows,1);
        for i=1:nrows   %each row is a different time; there may be only one 

            s = ess(i);                % flash location fed into FlashBC as global
            Tm2 = Tm(i,2); % leftmost temperature in magma, needed by FlashBC, global
            Tm3 = Tm(i,3);
            p2= pv2(i) ;   % needed by FlashBC, passed as global
            p3= pv3(i) ;   % needed by FlashBC, passed as global
            Tnp1 = Ts(i,n+1); % temperature in slurry next to flash, needed by FlashBC, global
            Tn = Ts(i, n); 
            % check straddle zero. Look between slurry temp and critical
            % temp
            %r1=FlashBC(T0);
            r1=FlashBC(0.25);
            r2=FlashBC(0.5); %T=0.5 is approximately critical temperature
            if r1*r2 > 0   % same sign, not straddling!!
                disp('not straddling zero; s,Tm2,p2,Tn+1:')
                disp(s)
                disp(Tm2)
                disp(p2)
                disp(Tnp1)

                Tflash(i) = fzero(@FlashBC,Tnp1); % look for a solution near the slurry T
                disp(Tflash(i))
                LookAtFlash
                disp(StopNow)
            else % we do have straddle:
            Tflash(i) = fzero(@FlashBC,[0.25 0.5]); % solve BC at flash for temperature there
            end
  
        end  
        
        Ts(:,n+2) = Tflash; Tm(:,1) = Tflash; % temperatures match at flash point
        
        % other values can now be computed at the flash point:
        Pflash = exp(Haitch*(Tflash - T0)./Tflash);
        % 
        rho(:,1) = Pflash./Tflash; 
        
        % values at surface of bomb:
         
        Tm(:,m+2) = Tsurface;  % set T of vapour at magma surface (to boiling point at one atm)  
        rho(:,m+2) = p0/Tsurface; % boundary value of vapour density  
        
    end 

    function RSide = SurtseyModel(~, y)
        %% SurtseyModel sets up the RHS of the differential equations
        %  RSide should be a column vector of same length as initial
        %  conditions. It doesn't depend on time, hence the ~.
        
        
        global epsilon4 epsilon5 
                                 
        % here are the actual variables in the ODEs:
        % unpack y to T, Tm, rho, s values (T slurry-n terms in y, T magma-m terms in y, density of
        % vapour in magma-m terms, location of flashing front-1 term). Add
        % two boundary values to Temperatures and density
        
        [T,Tm,rho,s] = UnpackSolutions(y); % This generates boundary values too
                                           % so LHS winds up with more
                                           % terms than y has      
        p = rho.*Tm;  % pressures in magma, the full set from 1 to m+2
%        
% the above pressure is correct at the flash point, where pressure is set
% by the clausius clapeyron formula. It is OK since I have set the value of
% rho here using a pressure from Clausius Clapeyron, anyway.

        %dsdt = epsilon4*Cm(1)*rho(1)/(1.0-s) * (p(2)-p(1)); % we need this now,
                                              % the rate of change of s
                                              % replace with a second-order
                                              % accurate one-sided
                                              % difference for dp/d chi
        dsdt = epsilon4*Cm(1)*rho(1)/(1.0-s) * (-3*p(1)+4*p(2)-p(3))/2;                                      
          
        % Now compute the RHsides of the differential equations:
        %
        % T in slurry: there are n terms plus 2 boundary values in the
        % slurry
        %
        
        vT = -(zeta(2:n+1).* zeta(2:n+1)*s*dsdt + 2*epsilon3)./(s*s*zeta(2:n+1));

        wT = vT.*Cn(2:n+1) - 2*aas*Bn*(Cn(2:n+1).*chi(2:n+1))*epsilon3/s/s;
        % distinguish between positive and negative velocity terms
        wp = max(wT, 0); wm = min(wT, 0);  % note that these work for vectors vT
        Tz = diff(T);  % T is of length n+2; Tz has length n+1
        Tzm = Tz(1:n); Tzp = Tz(2:n+1);
        Tzup = wp.*Tzm + wm.*Tzp;   % this gives upwind differencing on first deriv terms
        
        Tzz = diff(Tz);  % this is the second derivative wrt chi, since chi spacing is 1
        
        dTdt = -Tzup + epsilon3*Cn(2:n+1).*Cn(2:n+1) .* Tzz/s/s;
        
        % T in magma:  
        
        % use upwinding on the first spatial derivative terms:
        
        vTm = (xi(2:m+1)-1)*dsdt/(1.0-s) - 2*delta5./((1-s)*( s + (1-s)*xi(2:m+1)));
        WTm = vTm.*Cm(2:m+1) - 2*aam*Bm*Cm(2:m+1).*(psi(2:m+1)-m-1)*delta5/(1-s)/(1-s);
        vpm = max(WTm, 0); vmm = min(WTm, 0);  % note that these work for vectors vTm too
        Tmxi = diff(Tm) ;  % Tm is of length n+2; Tmxi has length n+1
        Txim = Tmxi(1:m); Txip = Tmxi(2:m+1);
        Tmup = vpm.*Txim + vmm.*Txip;  % does upwinding on d Tm/ d psi terms

        Tmzz = diff(Tmxi);  % second spatial derivative of Tm
        
        dTmdt = -Tmup + delta5*Cm(2:m+1).*Cm(2:m+1)/((1-s)*(1-s)) .* Tmzz;  %T in magma
        
        % now do the rho equation in the magma. xi is the spatial variable
        % in the magma:
        
        Vrhoi = (xi(2:m+1)-1).*Cm(2:m+1) * dsdt/(1-s);
                 
        vprho = max(Vrhoi, 0); vmrho= min(Vrhoi, 0);
        drho = diff(rho);        
        rhozm = drho(1:m); rhozp= drho(2:m+1);
        rhozup = vprho.*rhozm + vmrho.*rhozp;  % upwinding value of coeff * d rho/ dz
            
        VPrRho=-2*epsilon5./( (1-s)*(s+ (1-s)*xi(2:m+1)));  % the Vp part only
        VPrRho= VPrRho.*Cm(2:m+1)- epsilon5*2*aam*Bm*(Cm(2:m+1).*(psi(2:m+1)-m-1))/(1-s)^2;
        VPrRho = VPrRho.*rho(2:m+1);
        VPpos=max(VPrRho,0); VPneg=min(VPrRho,0);
        dpee = diff(p); % should be m+1 terms in here
        pneg=dpee(1:m); ppos=dpee(2:m+1);
        VPup=VPpos.*pneg + VPneg.*ppos;    % generates the rho*dp/d psi term in rho equation
        
        % a more accurate second difference term uses average rho values:
        
        Pflux = (rho(1:m+1) + rho(2:m+2)).* dpee/2; % the nonlinear flux term
        Peepp = diff(Pflux);
        
        dRhodt = -rhozup -VPup + ...
           epsilon5/( (1-s)*(1-s)) *Cm(2:m+1).*Cm(2:m+1).* Peepp;  % the RHS of density DE
        
        RSide = [dTdt'; dTmdt'; dRhodt'; dsdt];
    end
        

    
    function LookAtFlash
        %% LookAtFlash plots the LHS and RHS of the flash condition
        % this is exploratory
        % I find that for T in the range T0,0.5 (that is, two-phase
        % conditions), there is only one solution with initial conditions
        % in place. This changed however, when temperatures dropped in
        % magma and increased in slurry. The model was flawed though...


        T = linspace(T0/10,0.5,500);

        LHS = LeftSide(T);
        
        RHS = RightSide(T);
        
        
        figure(21)
        clf('reset')
        plot(T,LHS,'-k','LineWidth', 3)
        hold on;
        plot(T,RHS,'-r', 'LineWidth', 3)
        Taxis=min(min(LHS),min(RHS));
        MaxV=max(max(LHS),max(RHS));
        plot([Tnp1, Tnp1],[Taxis, MaxV],'-b')
        plot([Tm2, Tm2],[Taxis, MaxV],'-g')

        ax=gca;
        defFont=24;
        ax.FontSize=defFont; ax.FontName='Times';
        hold off;
        xlabel('T^*');

        figure(22)
        clf('reset')
        plot(T,log10(abs(LHS)),'-k','LineWidth', 3)
        hold on;
        plot(T,log10(abs(RHS)),'-r', 'LineWidth', 3)
        ax=gca;
        defFont=24;
        ax.FontSize=defFont; ax.FontName='Times';
        title('logs taken');
        hold off;
    end
    


    function result =FlashBC(T)
    %% FlashBC computes the balance across the flash
    % this balance will determine the value of temperature there
    %
    % variables passed as globals to LeftSide are p2 and p3, pressures
    % adjacent to flash, and variables passed as globals to RightSide are 
    % temperatures Tnp1 and Tn in slurry, 
    % and temperatures Tm2 and Tm3 in magma, all adjacent to flash
        
        LHS = LeftSide(T);
        
        RHS = RightSide(T);

        result=LHS-RHS;  %fzero will try to find when this value is zero
        
    end

    function LHS = LeftSide(T)
    %% LeftSide computes the left-hand side of the flash condition
        global epsilon4 Haitch St
        
        p=exp(Haitch*(T-T0)./T);
    
    % the following is a first-order difference approximation
    
    %   LHS = St*epsilon4*p.*(p2-p);
 % here is a second-order one-sided difference approximation to the gradient:
 % it requires an extra pressure value adjacent to flash, p3, as well as p2
        LHS = St*epsilon4*p.*(-p3+4*p2-3*p);
 
    end

    function RHS = RightSide(T)
    %% RightSide computes the right-hand side of the flash condition
    %  
        
    % RHS = T.*T/s + T*((s-1)/s *Tnp1 -Tm2);  % this is OK iff m=n
    
% the following formula is first-order accurate, and only requires the adjacent values
% Tm2, Tnp1  for temperature in magma and slurry
%
% RHS = -T.*(Tm2-T+Bn/Bm  * aam/aas *(s-1)/s * ...
%         (1+aas*aas*(n+1)*(n+1))/(1+aam*aam*(m+1)*(m+1)) *(T-Tnp1));  % allows m not equal to n; and aam not equal to aas (the value of a in the slurry)

% the following formula is a second-order one-sided difference. It requires
% values for the location s of the flashing front, and values of 
% Tm2, Tm3, Tnp1, Tn  for temperatures in magma and slurry
% adjacent to flash:

    RHS = -T.*(-Tm3 +4*Tm2-3*T +Bn/Bm  * aam/aas *(s-1)/s * ...
        (1+aas*aas*(n+1)*(n+1))/(1+aam*aam*(m+1)*(m+1)) *(3*T-4*Tnp1+Tn));  % allows m not equal to n; and aam not equal to aas (the value of a in the slurry)
               
    end

    function result = FlashBCinit(T)
    %% computes the flash condition for initial setup only, with error fn T similarity solns
    %  the similarity solution is used for T gradient in magma at next node 
    %  and for T gradient in slurry too.
    % tSince is time since emplacement, time since the similarity solution
    % was a step function
        
    global St epsilon4 Haitch
        
    p=exp(Haitch*(T-T0)./T);
    Cm0 = Bm/aam * (1+aam*aam*(m+1)*(m+1));
    LHS =  St*epsilon4* p *(-p3+4*p2-3*p);
    RHS = -2*T/(Cm0*sqrt(pi*tSince)) * (1-s0)*( (1-T)/sqrt(delta5) - (T-T0)/sqrt(epsilon3) );  % allows m not equal to n; and aam not equal to aas (the value of a in the slurry)
    
    
    result = LHS-RHS;           
    end

%     function result = LookAtFlashBCInit(T)
%     %% checks the computation of flash condition for initial setup only
%     %  the similarity solution is used for T gradient in magma at next node 
%     %  and for T gradient in slurry too.
%     % tSince is time since emplacement, time since the similarity solution
%     % was a step function
%         
%     global St epsilon4 Haitch
%         
%     p=exp(Haitch*(T-T0)./T);
%     Cm0 = Bm/aam * (1+aam*aam*(m+1)*(m+1));
%     LHS =  St*epsilon4* p *(-p3+4*p2-3*p);
%     A1 = 2/(Cm0*sqrt(pi*tSince)) * (1-s0)*( (1-T)/sqrt(delta5));  
%     A2 = 2/(Cm0*sqrt(pi*tSince)) * (1-s0)*( (T-T0)/sqrt(epsilon3) );  
%      
%     result = [LHS, A1, A2];           
%     end

function  RescaleSurtsey
%% RescaleSurtsey does the rescalings that nondimensionalise the full P,T
% model of steaming surtseyan bombs, and computes parameter values
% 

global Haitch epsilon4 epsilon5 St Tsurface

R2 = 0.1; %bomb size, m
R1 = 0.01; % inclusion size, m
s0=R1/R2;
Tm = 1275.0; % magma temperature, K
Pa = 1.0E05; %scale pressure on atmospheric
Ti = 373; % initial temperature of inclusion, K

rhom = 2750.0;  %density of magma ignoring pores; multipy by porosity to get apparent density
Cpm =  840.0;   %thermal capacity of magma rock, J/kg/K
Cpl = 4200;  %thermal capacity of liquid water, J/kg/K
Cps = 2000; % thermal capacity of water vapour, J/kg/K
Ke=2;    % thermal conductivity, gas filled magma
Kel=3;    % thermal conductivity, liquid filled magma
M=18.0e-03; %molar mass water, kg/mol

%phi = 0.4; %magma porosity
Hsl = 2.3e06; % specific heat of vaporisation for water, J/kg

R=8.314;   % universal gas constant, J/kg/mol

rho_0l = 1000; % reference liquid water density, kg/m^3
muv = 3e-05; % dynamic viscosity Pa.s

Haitch = M*Hsl/(R*Ti);
rhos=Pa*M/(R*Tm);  % called \rho_{0s} in notes
rockheat=(1-phi)*rhom*Cpm;
rscs=phi*rhos*Cps;
rhoc=rockheat + rscs;
rhopCp=rockheat+ phi*rho_0l*Cpl;

t0=phi*rho_0l*Hsl*R1*R1/(3*Ke*(Tm-Ti));
v0s=R2*rho_0l/(t0*rhos);

delta5=Ke*t0/(rhoc*R2*R2);

epsilon3 = Kel*t0/(R2*R2*rhopCp);
St=Hsl*phi*rhos*R2*v0s/(Tm*Ke);
T0=Ti/Tm; % initial temperature in the slurry; boiling point at one atm
Tsurface = T0; % set surface T of magma to boiling point for one atmosphere, and rescale it
epsilon4=k*Pa/(muv*R2*phi*v0s);
epsilon5 = k*Pa*t0/(phi*muv*R2*R2);


% theoretical pmax value:

RampDistance=1E-05/R2; % pore diameter, 10 microns before normalising

% Ap = 3*Ke*muv*sqrt(rhoc*rho_0l)/(pi*sqrt(3)*RampDistance*Pa*Hsl^(3/2)*rhos*rhos);
% pmaxcubed = Ap/(k*sqrt(phi))  * Tm*Tm/(sqrt(Tm-Ti)) * R1*R1/R2/R2 * (R2-R1)/R2;
% pmaxtheor = pmaxcubed^(1/3);

pmaxtheor = sqrt(8*s0*(1-s0)*(1-T0)*Tm*muv*Ke/(k*Pa*Hsl*rhos*sqrt(pi)*RampDistance));

display(pmaxtheor)
pmaxtheorCompare = sqrt(8*s0*(1-s0)*(1-T0)/(epsilon4*St*sqrt(pi)*RampDistance));
display(pmaxtheorCompare)

% theoretical time at which this max pressure is reached:

BeeSQ=1.2*s0*(1-s0)/(epsilon4*St*sqrt(pi*delta5));
Bee2 = 0.3* sqrt(epsilon5)/(epsilon4*St*sqrt(pi*delta5));
tmaxtheor = BeeSQ/(4*Bee2*Bee2) * RampDistance/sqrt(delta5);

% get a more accurate value of tmaxtheor and pmaxtheor:
       
tee = RampDistance^2 * delta5/ 16;
% display(tee)
% 
display(tmaxtheor)

Value1 = PsCross(tmaxtheor*1E-08);
disp(Value1)

Value3 = PsCross(tmaxtheor*1E03);
disp(Value3)



tmaxtheorACC = fzero(@PsCross, tmaxtheor*[1E-08, 1E03]);
pmaxtheorACC = sqrt(BeeSQ)/((tmaxtheorACC + tee)^(1/4));

    function value=PsCross(t)
       %% to calculate the crossing time when early pressure solution crosses nullcline
       % uses a less approximate version of these solutions for a more
       % accurate crossing time than the asymptotic value as tstar goes to
       % zero
       Pe = 1.0 + Bee2*(log(2*sqrt(t.^2 + t*tee) + 2*t + tee) - log(tee));
       Pqu = sqrt(BeeSQ)/((t + tee)^(1/4));
       value = Pe-Pqu;     % fzero will try to make this zero so the pressures are equal
    end


end

    function y = GetZmag(a) 
        %% the zero of this function determines a in magma
        % and the resulting a gives the correct minimum mesh in magma
       y = 1 - atan(a*m)/atan(a*(m+1)) - ximin; 
    end


%     function y = GetZslurry(en) 
%         %% the zero of this function determines n given aam
%         % and the resulting n gives almost the correct minimum mesh in slurry
%        y = 1 - atan(aam*en)/atan(aam*(en+1)) - zetamin; 
%     end

    function y = GetAAslurry(a) 
        %% the zero of this function determines a in slurry, given n
        % and the resulting a gives the correct minimum mesh in slurry
       y = 1 - atan(a*n)/atan(a*(n+1)) - zetamin; 
    end

    function [val,ist,dir]=MyEvents(t,y)
        %% to stop if s gets too small or max p is reached
        val1=y(end)-1.0E-03;  % this is s-smin. dimensionless s
        ist1=1;         %this is terminal; stop if encountered
        dir1=0;         %any direction
        % also stop if max temperature is reached at flash. Use the temperature
        % in the magma as a proxy for this:
        RightSide = SurtseyModel(t,y); % generate the right-hand sides of
                                       % the ODEs
        rhodot=RightSide(n+m+1); % time rate of change of the first rho in magma
                              % next to the flash. This reaches a max at a
                              % time close to and just after when Pflash
                              % does
        Tdot=RightSide(n+1); % time rate of change of T in magma next to flash
        
        % P is rho* T; use product rule to find time rate of change of P in
        % magma:
        T2=y(n+1); Rho2=y(n+m+1); % need actual values too
        val2 = rhodot*T2 + Rho2*Tdot; % rate of change of pressure at first node in magma, next to flash

        ist2=1; % stop if encounter this being zero
        dir2=-1;    % direction is decreasing dT/dt for a maximum of T there
        
        val=[val1; val2];
        ist=[ist1; ist2];
        dir=[dir1; dir2];


    end
end

